/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.liftover;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.liftover.Chain;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.OverlapDetector;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class LiftOver {
    private static final Log LOG = Log.getInstance(LiftOver.class);
    public static final double DEFAULT_LIFTOVER_MINMATCH = 0.95;
    private double liftOverMinMatch = 0.95;
    private final OverlapDetector<Chain> chains;
    private final Map<String, Set<String>> contigMap = new HashMap<String, Set<String>>();
    private boolean logFailedIntervals = true;
    private long totalFailedIntervalsBelowThreshold = 0L;

    public void setShouldLogFailedIntervalsBelowThreshold(boolean logFailedIntervals) {
        this.logFailedIntervals = logFailedIntervals;
    }

    public void resetFailedIntervalsBelowThresholdCounter() {
        this.totalFailedIntervalsBelowThreshold = 0L;
    }

    public long getFailedIntervalsBelowThreshold() {
        return this.totalFailedIntervalsBelowThreshold;
    }

    public LiftOver(File chainFile) {
        IOUtil.assertFileIsReadable(chainFile);
        this.chains = Chain.loadChains(chainFile);
        for (Chain chain : this.chains.getAll()) {
            Set<Object> names;
            String from = chain.fromSequenceName;
            String to = chain.toSequenceName;
            if (this.contigMap.containsKey(from)) {
                names = this.contigMap.get(from);
            } else {
                names = new HashSet();
                this.contigMap.put(from, names);
            }
            names.add(to);
        }
    }

    public void validateToSequences(SAMSequenceDictionary sequenceDictionary) {
        for (Chain chain : this.chains.getAll()) {
            if (sequenceDictionary.getSequence(chain.toSequenceName) != null) continue;
            throw new SAMException("Sequence " + chain.toSequenceName + " from chain file is not found in sequence dictionary.");
        }
    }

    public Interval liftOver(Interval interval) {
        return this.liftOver(interval, this.liftOverMinMatch);
    }

    public Interval liftOver(Interval interval, double liftOverMinMatch) {
        if (interval.length() == 0) {
            throw new IllegalArgumentException("Zero-length interval cannot be lifted over.  Interval: " + interval.getName());
        }
        Chain chainHit = null;
        TargetIntersection targetIntersection = null;
        double minMatchSize = liftOverMinMatch * (double)interval.length();
        boolean hasOverlapBelowThreshold = false;
        for (Chain chain : this.chains.getOverlaps(interval)) {
            TargetIntersection candidateIntersection = LiftOver.targetIntersection(chain, interval);
            if (candidateIntersection != null && (double)candidateIntersection.intersectionLength >= minMatchSize) {
                if (chainHit != null) {
                    return null;
                }
                chainHit = chain;
                targetIntersection = candidateIntersection;
                continue;
            }
            if (candidateIntersection == null) continue;
            hasOverlapBelowThreshold = true;
            if (!this.logFailedIntervals) continue;
            LOG.info("Interval " + interval.getName() + " failed to match chain " + chain.id + " because intersection length " + candidateIntersection.intersectionLength + " < minMatchSize " + minMatchSize + " (" + (float)candidateIntersection.intersectionLength / (float)interval.length() + " < " + liftOverMinMatch + ")");
        }
        if (chainHit == null) {
            if (hasOverlapBelowThreshold) {
                ++this.totalFailedIntervalsBelowThreshold;
            }
            return null;
        }
        return LiftOver.createToInterval(interval.getName(), interval.isNegativeStrand(), targetIntersection);
    }

    public List<PartialLiftover> diagnosticLiftover(Interval interval) {
        ArrayList<PartialLiftover> ret = new ArrayList<PartialLiftover>();
        if (interval.length() == 0) {
            throw new IllegalArgumentException("Zero-length interval cannot be lifted over.  Interval: " + interval.getName());
        }
        for (Chain chain : this.chains.getOverlaps(interval)) {
            Interval intersectingChain;
            TargetIntersection targetIntersection = LiftOver.targetIntersection(chain, intersectingChain = interval.intersect(chain.interval));
            if (targetIntersection == null) {
                ret.add(new PartialLiftover(intersectingChain, chain.id));
                continue;
            }
            Interval toInterval = LiftOver.createToInterval(interval.getName(), interval.isNegativeStrand(), targetIntersection);
            float percentLiftedOver = (float)targetIntersection.intersectionLength / (float)interval.length();
            ret.add(new PartialLiftover(intersectingChain, toInterval, targetIntersection.chain.id, percentLiftedOver));
        }
        return ret;
    }

    public Map<String, Set<String>> getContigMap() {
        return Collections.unmodifiableMap(this.contigMap);
    }

    private static Interval createToInterval(String intervalName, boolean sourceNegativeStrand, TargetIntersection targetIntersection) {
        int toStart = targetIntersection.chain.getBlock((int)targetIntersection.firstBlockIndex).toStart + targetIntersection.startOffset;
        int toEnd = targetIntersection.chain.getBlock(targetIntersection.lastBlockIndex).getToEnd() - targetIntersection.offsetFromEnd;
        if (toEnd <= toStart || toStart < 0) {
            throw new SAMException("Something strange lifting over interval " + intervalName);
        }
        if (targetIntersection.chain.toOppositeStrand) {
            int negativeStart = targetIntersection.chain.toSequenceSize - toEnd;
            int negativeEnd = targetIntersection.chain.toSequenceSize - toStart;
            toStart = negativeStart;
            toEnd = negativeEnd;
        }
        boolean negativeStrand = targetIntersection.chain.toOppositeStrand ? !sourceNegativeStrand : sourceNegativeStrand;
        return new Interval(targetIntersection.chain.toSequenceName, toStart + 1, toEnd, negativeStrand, intervalName);
    }

    private static TargetIntersection targetIntersection(Chain chain, Interval interval) {
        int intersectionLength = 0;
        int start = interval.getStart() - 1;
        int end = interval.getEnd();
        int firstBlockIndex = -1;
        int lastBlockIndex = -1;
        int startOffset = -1;
        int offsetFromEnd = -1;
        List<Chain.ContinuousBlock> blockList = chain.getBlocks();
        int i = 0;
        while (i < blockList.size()) {
            Chain.ContinuousBlock block = blockList.get(i);
            if (block.fromStart >= end) break;
            if (block.getFromEnd() > start) {
                if (firstBlockIndex == -1) {
                    firstBlockIndex = i;
                    startOffset = start > block.fromStart ? start - block.fromStart : 0;
                }
                lastBlockIndex = i;
                offsetFromEnd = block.getFromEnd() > end ? block.getFromEnd() - end : 0;
                int thisIntersection = Math.min(end, block.getFromEnd()) - Math.max(start, block.fromStart);
                if (thisIntersection <= 0) {
                    throw new SAMException("Should have been some intersection.");
                }
                intersectionLength += thisIntersection;
            }
            ++i;
        }
        if (intersectionLength == 0) {
            return null;
        }
        return new TargetIntersection(chain, intersectionLength, startOffset, offsetFromEnd, firstBlockIndex, lastBlockIndex);
    }

    public double getLiftOverMinMatch() {
        return this.liftOverMinMatch;
    }

    public void setLiftOverMinMatch(double liftOverMinMatch) {
        this.liftOverMinMatch = liftOverMinMatch;
    }

    public static class PartialLiftover {
        final Interval fromInterval;
        final Interval toInterval;
        final int chainId;
        final float percentLiftedOver;

        PartialLiftover(Interval fromInterval, Interval toInterval, int chainId, float percentLiftedOver) {
            this.fromInterval = fromInterval;
            this.toInterval = toInterval;
            this.chainId = chainId;
            this.percentLiftedOver = percentLiftedOver;
        }

        PartialLiftover(Interval fromInterval, int chainId) {
            this.fromInterval = fromInterval;
            this.toInterval = null;
            this.chainId = chainId;
            this.percentLiftedOver = 0.0f;
        }

        public String toString() {
            if (this.toInterval == null) {
                return String.valueOf(this.fromInterval.toString()) + " (len " + this.fromInterval.length() + ")=>null using chain " + this.chainId;
            }
            String strand = this.toInterval.getStrand().encode();
            return String.valueOf(this.fromInterval.toString()) + " (len " + this.fromInterval.length() + ")=>" + this.toInterval + "(" + strand + ") using chain " + this.chainId + " ; pct matched " + this.percentLiftedOver;
        }
    }

    private static class TargetIntersection {
        final Chain chain;
        final int intersectionLength;
        final int startOffset;
        final int offsetFromEnd;
        final int firstBlockIndex;
        final int lastBlockIndex;

        TargetIntersection(Chain chain, int intersectionLength, int startOffset, int offsetFromEnd, int firstBlockIndex, int lastBlockIndex) {
            this.chain = chain;
            this.intersectionLength = intersectionLength;
            this.startOffset = startOffset;
            this.offsetFromEnd = offsetFromEnd;
            this.firstBlockIndex = firstBlockIndex;
            this.lastBlockIndex = lastBlockIndex;
        }
    }
}

