/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.tribble.index.tabix;

import htsjdk.samtools.Bin;
import htsjdk.samtools.BinningIndexContent;
import htsjdk.samtools.Chunk;
import htsjdk.samtools.LinearIndex;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.StringUtil;
import htsjdk.tribble.Tribble;
import htsjdk.tribble.TribbleException;
import htsjdk.tribble.index.Block;
import htsjdk.tribble.index.Index;
import htsjdk.tribble.index.tabix.TabixFormat;
import htsjdk.tribble.util.LittleEndianInputStream;
import htsjdk.tribble.util.LittleEndianOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class TabixIndex
implements Index {
    private static final byte[] MAGIC = new byte[]{84, 66, 73, 1};
    public static final int MAGIC_NUMBER;
    private final TabixFormat formatSpec;
    private final List<String> sequenceNames;
    private final BinningIndexContent[] indices;

    static {
        ByteBuffer bb = ByteBuffer.allocate(MAGIC.length);
        bb.put(MAGIC);
        bb.flip();
        MAGIC_NUMBER = bb.order(ByteOrder.LITTLE_ENDIAN).getInt();
    }

    public TabixIndex(TabixFormat formatSpec, List<String> sequenceNames, BinningIndexContent[] indices) {
        if (sequenceNames.size() != indices.length) {
            throw new IllegalArgumentException("sequenceNames.size() != indices.length");
        }
        this.formatSpec = formatSpec.clone();
        this.sequenceNames = Collections.unmodifiableList(new ArrayList<String>(sequenceNames));
        this.indices = indices;
    }

    public TabixIndex(InputStream inputStream) throws IOException {
        this(inputStream, false);
    }

    public TabixIndex(File tabixFile) throws IOException {
        this(new BlockCompressedInputStream(tabixFile), true);
    }

    public TabixIndex(Path tabixPath) throws IOException {
        this(new BlockCompressedInputStream(Files.newInputStream(tabixPath, new OpenOption[0])), true);
    }

    private TabixIndex(InputStream inputStream, boolean closeInputStream) throws IOException {
        LittleEndianInputStream dis = new LittleEndianInputStream(inputStream);
        if (dis.readInt() != MAGIC_NUMBER) {
            throw new TribbleException(String.format("Unexpected magic number 0x%x", MAGIC_NUMBER));
        }
        int numSequences = dis.readInt();
        this.indices = new BinningIndexContent[numSequences];
        this.formatSpec = new TabixFormat();
        this.formatSpec.flags = dis.readInt();
        this.formatSpec.sequenceColumn = dis.readInt();
        this.formatSpec.startPositionColumn = dis.readInt();
        this.formatSpec.endPositionColumn = dis.readInt();
        this.formatSpec.metaCharacter = (char)dis.readInt();
        this.formatSpec.numHeaderLinesToSkip = dis.readInt();
        int nameBlockSize = dis.readInt();
        byte[] nameBlock = new byte[nameBlockSize];
        if (dis.read(nameBlock) != nameBlockSize) {
            throw new EOFException("Premature end of file reading Tabix header");
        }
        ArrayList<String> sequenceNames = new ArrayList<String>(numSequences);
        int startPos = 0;
        int i = 0;
        while (i < numSequences) {
            int endPos = startPos;
            while (nameBlock[endPos] != 0) {
                ++endPos;
            }
            sequenceNames.add(StringUtil.bytesToString(nameBlock, startPos, endPos - startPos));
            startPos = endPos + 1;
            ++i;
        }
        if (startPos != nameBlockSize) {
            throw new TribbleException("Tabix header format exception.  Sequence name block is longer than expected");
        }
        i = 0;
        while (i < numSequences) {
            this.indices[i] = this.loadSequence(i, dis);
            ++i;
        }
        if (closeInputStream) {
            CloserUtil.close(dis);
        }
        this.sequenceNames = Collections.unmodifiableList(sequenceNames);
    }

    @Override
    public List<Block> getBlocks(String chr, int start, int end) {
        int sequenceIndex = this.sequenceNames.indexOf(chr);
        if (sequenceIndex == -1 || this.indices[sequenceIndex] == null) {
            return Collections.emptyList();
        }
        List<Chunk> chunks = this.indices[sequenceIndex].getChunksOverlapping(start, end);
        if (chunks == null) {
            return Collections.emptyList();
        }
        ArrayList<Block> ret = new ArrayList<Block>(chunks.size());
        chunks.stream().map(chunk -> new Block(chunk.getChunkStart(), chunk.getChunkEnd() - chunk.getChunkStart())).forEach(ret::add);
        return ret;
    }

    @Override
    public boolean isCurrentVersion() {
        return true;
    }

    @Override
    public List<String> getSequenceNames() {
        return this.sequenceNames;
    }

    @Override
    public boolean containsChromosome(String chr) {
        return this.sequenceNames.contains(chr);
    }

    @Override
    public Map<String, String> getProperties() {
        return null;
    }

    @Override
    public boolean equalsIgnoreProperties(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TabixIndex that = (TabixIndex)o;
        if (!this.formatSpec.equals(that.formatSpec)) {
            return false;
        }
        if (!Arrays.equals(this.indices, that.indices)) {
            return false;
        }
        return this.sequenceNames.equals(that.sequenceNames);
    }

    public TabixFormat getFormatSpec() {
        return this.formatSpec;
    }

    @Override
    public void write(Path tabixPath) throws IOException {
        Throwable throwable = null;
        Object var3_4 = null;
        try (LittleEndianOutputStream los = new LittleEndianOutputStream(new BlockCompressedOutputStream(Files.newOutputStream(tabixPath, new OpenOption[0]), null));){
            this.write(los);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    @Override
    public void writeBasedOnFeaturePath(Path featurePath) throws IOException {
        if (!Files.isRegularFile(featurePath, new LinkOption[0])) {
            throw new IOException("Cannot write based on a non-regular file: " + featurePath.toUri());
        }
        this.write(Tribble.tabixIndexPath(featurePath));
    }

    @Override
    public void write(LittleEndianOutputStream los) throws IOException {
        los.writeInt(MAGIC_NUMBER);
        los.writeInt(this.sequenceNames.size());
        los.writeInt(this.formatSpec.flags);
        los.writeInt(this.formatSpec.sequenceColumn);
        los.writeInt(this.formatSpec.startPositionColumn);
        los.writeInt(this.formatSpec.endPositionColumn);
        los.writeInt(this.formatSpec.metaCharacter);
        los.writeInt(this.formatSpec.numHeaderLinesToSkip);
        int nameBlockSize = this.sequenceNames.size();
        for (String sequenceName : this.sequenceNames) {
            nameBlockSize += sequenceName.length();
        }
        los.writeInt(nameBlockSize);
        for (String sequenceName : this.sequenceNames) {
            los.write(StringUtil.stringToBytes(sequenceName));
            los.write(0);
        }
        BinningIndexContent[] binningIndexContentArray = this.indices;
        int n = this.indices.length;
        int n2 = 0;
        while (n2 < n) {
            BinningIndexContent index = binningIndexContentArray[n2];
            this.writeSequence(index, los);
            ++n2;
        }
    }

    private void writeSequence(BinningIndexContent indexContent, LittleEndianOutputStream los) throws IOException {
        if (indexContent == null) {
            los.writeInt(0);
        } else {
            BinningIndexContent.BinList binList = indexContent.getBins();
            los.writeInt(binList.numberOfNonNullBins);
            for (Bin bin : binList) {
                this.writeBin(bin, los);
            }
            this.writeLinearIndex(indexContent.getLinearIndex(), los);
        }
    }

    private void writeLinearIndex(LinearIndex linearIndex, LittleEndianOutputStream los) throws IOException {
        if (linearIndex.getIndexStart() != 0) {
            throw new IllegalArgumentException("Non-zero linear index start");
        }
        long[] entries = linearIndex.getIndexEntries();
        los.writeInt(entries.length);
        long[] lArray = entries;
        int n = entries.length;
        int n2 = 0;
        while (n2 < n) {
            long entry = lArray[n2];
            los.writeLong(entry);
            ++n2;
        }
    }

    private void writeBin(Bin bin, LittleEndianOutputStream los) throws IOException {
        los.writeInt(bin.getBinNumber());
        List<Chunk> chunkList = bin.getChunkList();
        los.writeInt(chunkList.size());
        for (Chunk chunk : chunkList) {
            los.writeLong(chunk.getChunkStart());
            los.writeLong(chunk.getChunkEnd());
        }
    }

    private BinningIndexContent loadSequence(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int numBins = dis.readInt();
        if (numBins == 0) {
            return null;
        }
        int nonNullBins = 0;
        ArrayList<Bin> bins = new ArrayList<Bin>();
        int i = 0;
        while (i < numBins) {
            Bin bin = this.loadBin(referenceSequenceIndex, dis);
            if (bin != null) {
                ++nonNullBins;
                if (bins.size() > bin.getBinNumber()) {
                    if (bins.get(bin.getBinNumber()) != null) {
                        throw new TribbleException("Bin " + bin.getBinNumber() + " appears more than once in file");
                    }
                    bins.set(bin.getBinNumber(), bin);
                } else {
                    bins.ensureCapacity(bin.getBinNumber() + 1);
                    while (bins.size() < bin.getBinNumber()) {
                        bins.add(null);
                    }
                    bins.add(bin);
                }
            }
            ++i;
        }
        LinearIndex linearIndex = this.loadLinearIndex(referenceSequenceIndex, dis);
        return new BinningIndexContent(referenceSequenceIndex, new BinningIndexContent.BinList(bins.toArray(new Bin[bins.size()]), nonNullBins), linearIndex);
    }

    private LinearIndex loadLinearIndex(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int numElements = dis.readInt();
        long[] elements = new long[numElements];
        int i = 0;
        while (i < numElements) {
            elements[i] = dis.readLong();
            ++i;
        }
        return new LinearIndex(referenceSequenceIndex, 0, elements);
    }

    private Bin loadBin(int referenceSequenceIndex, LittleEndianInputStream dis) throws IOException {
        int binNumber = dis.readInt();
        Bin ret = new Bin(referenceSequenceIndex, binNumber);
        int numChunks = dis.readInt();
        ArrayList<Chunk> chunkList = new ArrayList<Chunk>(numChunks);
        int i = 0;
        while (i < numChunks) {
            chunkList.add(this.loadChunk(dis));
            ++i;
        }
        ret.setChunkList(chunkList);
        return ret;
    }

    private Chunk loadChunk(LittleEndianInputStream dis) throws IOException {
        long start = dis.readLong();
        long end = dis.readLong();
        return new Chunk(start, end);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TabixIndex index = (TabixIndex)o;
        if (!this.formatSpec.equals(index.formatSpec)) {
            return false;
        }
        if (!Arrays.equals(this.indices, index.indices)) {
            return false;
        }
        return this.sequenceNames.equals(index.sequenceNames);
    }

    public int hashCode() {
        int result = this.formatSpec.hashCode();
        result = 31 * result + this.sequenceNames.hashCode();
        result = 31 * result + Arrays.hashCode(this.indices);
        return result;
    }
}

